package jnpf.base.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import jnpf.base.UserInfo;
import jnpf.base.entity.SystemEntity;
import jnpf.base.mapper.SystemMapper;
import jnpf.base.model.AppAuthorizationModel;
import jnpf.base.service.ModuleService;
import jnpf.base.service.SuperServiceImpl;
import jnpf.base.service.SystemService;
import jnpf.constant.CodeConst;
import jnpf.constant.PermissionConst;
import jnpf.permission.entity.RoleEntity;
import jnpf.permission.entity.UserEntity;
import jnpf.permission.model.user.WorkHandoverModel;
import jnpf.permission.service.CodeNumService;
import jnpf.permission.service.RoleService;
import jnpf.permission.service.UserService;
import jnpf.util.RandomUtil;
import jnpf.util.StringUtil;
import jnpf.util.UserProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;

/**
 * 系统
 *
 * @author JNPF开发平台组
 * @version V3.1.0
 * @copyright 引迈信息技术有限公司
 * @date 2019年9月27日 上午9:18
 */
@Service
public class SystemServiceImpl extends SuperServiceImpl<SystemMapper, SystemEntity> implements SystemService {
    @Autowired
    private ModuleService moduleService;
    @Autowired
    private RoleService roleService;
    @Autowired
    private UserService userService;
    @Autowired
    private CodeNumService codeNumService;

    @Override
    public List<SystemEntity> getList() {
        QueryWrapper<SystemEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(SystemEntity::getEnabledMark, 1);
        queryWrapper.lambda().orderByAsc(SystemEntity::getSortCode)
                .orderByDesc(SystemEntity::getCreatorTime);
        return this.list(queryWrapper);
    }

    @Override
    public List<SystemEntity> getList(String keyword, Boolean filterEnableMark, boolean verifyAuth, Boolean filterMain, boolean isList, List<String> moduleAuthorize) {
        UserInfo user = UserProvider.getUser();

        boolean flag = false;
        QueryWrapper<SystemEntity> queryWrapper = new QueryWrapper<>();
        if (StringUtil.isNotEmpty(keyword)) {
            flag = true;
            queryWrapper.lambda().and(t ->
                    t.like(SystemEntity::getFullName, keyword).or().like(SystemEntity::getEnCode, keyword)
                            .or().like(SystemEntity::getDescription, keyword)
            );
        }
        if (filterEnableMark == null) {
            queryWrapper.lambda().eq(SystemEntity::getEnabledMark, 0);
        } else if (filterEnableMark) {
            queryWrapper.lambda().eq(SystemEntity::getEnabledMark, 1);
        }

        // 过滤掉系统应用
        if (filterMain != null && filterMain) {
            queryWrapper.lambda().ne(SystemEntity::getIsMain, 1);
        }
        //判断权限列表
        if (!user.getIsAdministrator() && verifyAuth) {
            if (user.getIsDevRole()) {
                queryWrapper.lambda().and(t -> t
                        .eq(SystemEntity::getUserId, user.getUserId()).or()
                        .like(SystemEntity::getAuthorizeId, user.getUserId()).or()
                        .eq(SystemEntity::getAuthorizeId, PermissionConst.ALL_DEV_USER));
            } else {
                queryWrapper.lambda().eq(SystemEntity::getUserId, user.getUserId());
            }
        }

        //过滤租户分配黑名单
        if (moduleAuthorize.size() > 0) {
            queryWrapper.lambda().notIn(SystemEntity::getId, moduleAuthorize);
        }
        if (flag) {
            queryWrapper.lambda().orderByDesc(SystemEntity::getLastModifyTime);
        } else {
            queryWrapper.lambda().orderByAsc(SystemEntity::getSortCode).orderByDesc(SystemEntity::getCreatorTime);
        }

        return this.list(queryWrapper);
    }

    @Override
    public List<SystemEntity> getListByIdsKey(List<String> ids, String keyword) {
        if (CollectionUtil.isEmpty(ids)) return Collections.EMPTY_LIST;
        QueryWrapper<SystemEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().in(SystemEntity::getId, ids);
        boolean flag = false;
        if (StringUtil.isNotEmpty(keyword)) {
            flag = true;
            queryWrapper.lambda().and(t ->
                    t.like(SystemEntity::getFullName, keyword).or().like(SystemEntity::getEnCode, keyword)
                            .or().like(SystemEntity::getDescription, keyword)
            );
        }
        if (flag) {
            queryWrapper.lambda().orderByDesc(SystemEntity::getLastModifyTime);
        } else {
            queryWrapper.lambda().orderByAsc(SystemEntity::getSortCode).orderByDesc(SystemEntity::getCreatorTime);
        }
        return this.list(queryWrapper);
    }

    @Override
    public SystemEntity getInfo(String id) {
        QueryWrapper<SystemEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(SystemEntity::getId, id);
        return this.getOne(queryWrapper);
    }

    @Override
    public Boolean isExistFullName(String id, String fullName) {
        if (StringUtil.isEmpty(fullName)) return false;
        QueryWrapper<SystemEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(SystemEntity::getFullName, fullName);
        if (StringUtil.isNotEmpty(id)) {
            queryWrapper.lambda().ne(SystemEntity::getId, id);
        }
        return this.count(queryWrapper) > 0;
    }

    @Override
    public Boolean isExistEnCode(String id, String enCode) {
        if (StringUtil.isEmpty(enCode)) return false;
        QueryWrapper<SystemEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(SystemEntity::getEnCode, enCode);
        if (StringUtil.isNotEmpty(id)) {
            queryWrapper.lambda().ne(SystemEntity::getId, id);
        }
        return this.count(queryWrapper) > 0;
    }

    @Override
    @Transactional
    public Boolean create(SystemEntity entity) {
        String userId = UserProvider.getUser().getUserId();
        entity.setId(RandomUtil.uuId());
        if (StringUtil.isEmpty(entity.getEnCode())) {
            entity.setEnCode(codeNumService.getCodeFunction(() -> codeNumService.getCodeOnce(CodeConst.YY), code -> this.isExistEnCode(null, code)));
        }
        entity.setIsMain(0);
        entity.setCreatorUserId(userId);
        entity.setCreatorTime(new Date());
        entity.setUserId(userId);
        boolean save = this.save(entity);
        //创建审批中心菜单
        moduleService.createWorkMenu(entity.getId());
        return save;
    }

    @Override
    @Transactional
    public Boolean update(String id, SystemEntity entity) {
        entity.setId(id);
        if (entity.getIsMain() == null) {
            entity.setIsMain(0);
        }
        if (StringUtil.isEmpty(entity.getEnCode())) {
            entity.setEnCode(codeNumService.getCodeFunction(() -> codeNumService.getCodeOnce(CodeConst.YY), code -> this.isExistEnCode(id, code)));
        }
        entity.setLastModifyUserId(UserProvider.getUser().getUserId());
        entity.setLastModifyTime(new Date());
        return this.updateById(entity);
    }

    @Override
    @Transactional
    public Boolean delete(String id) {
        moduleService.deleteBySystemId(id);
        return this.removeById(id);
    }

    @Override
    public List<SystemEntity> getListByIds(List<String> list, List<String> moduleAuthorize) {
        List<SystemEntity> systemList = new ArrayList<>(16);
        if (list.size() > 0) {
            QueryWrapper<SystemEntity> queryWrapper = new QueryWrapper<>();
            if (moduleAuthorize != null && moduleAuthorize.size() > 0) {
                queryWrapper.lambda().notIn(SystemEntity::getId, moduleAuthorize);
            }
            queryWrapper.lambda().in(SystemEntity::getId, list);
            queryWrapper.lambda().eq(SystemEntity::getEnabledMark, 1);
            queryWrapper.lambda().orderByAsc(SystemEntity::getSortCode).orderByDesc(SystemEntity::getCreatorTime);
            return this.list(queryWrapper);
        }
        return systemList;
    }

    @Override
    public SystemEntity getInfoByEnCode(String enCode) {
        if (StringUtil.isEmpty(enCode)) {
            return null;
        }
        QueryWrapper<SystemEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(SystemEntity::getEnCode, enCode);
        return this.getOne(queryWrapper);
    }

    @Override
    public List<SystemEntity> findSystemAdmin(List<String> moduleAuthorize) {
        QueryWrapper<SystemEntity> queryWrapper = new QueryWrapper<>();
        if (moduleAuthorize != null && moduleAuthorize.size() > 0) {
            queryWrapper.lambda().notIn(SystemEntity::getId, moduleAuthorize);
        }
        queryWrapper.lambda().orderByAsc(SystemEntity::getSortCode).orderByDesc(SystemEntity::getCreatorTime);
        return this.list(queryWrapper);
    }

    @Override
    public boolean saveSystemAuthorizion(AppAuthorizationModel model) {
        SystemEntity info = this.getInfo(model.getSystemId());
        List<String> devUsers = model.getDevUsers();
        if (devUsers != null && !devUsers.isEmpty()) {
            StringBuilder stringBuilder = new StringBuilder();
            devUsers.forEach(item -> stringBuilder.append(item).append(","));
            info.setAuthorizeId(stringBuilder.toString().substring(0, stringBuilder.toString().length() - 1));
        } else {
            info.setAuthorizeId("");
        }

        if (model.getIsAllDevUser() != null && model.getIsAllDevUser() == 1) {
            info.setAuthorizeId(PermissionConst.ALL_DEV_USER);
        }
        return this.updateById(info);
    }

    @Override
    public List<SystemEntity> getAuthListByUser(String userId, Boolean isStand) {
        UserEntity user = userService.getInfo(userId);
        List<RoleEntity> userRoles = roleService.getUserRoles(userId);
        boolean isDevRole = userRoles.stream().anyMatch(t -> PermissionConst.DEVELOPER_CODE.equals(t.getEnCode()));
        if(isStand){
            isDevRole = UserProvider.getUser().getIsDevRole();
        }
        boolean isAdmin = Objects.equals(user.getIsAdministrator(), 1);

        QueryWrapper<SystemEntity> queryWrapper = new QueryWrapper<>();
        //开发人员才有编辑权限
        if (isDevRole || isAdmin) {
            //判断权限列表
            if (!isAdmin) {
                queryWrapper.lambda().eq(SystemEntity::getUserId, userId).or();
                queryWrapper.lambda().like(SystemEntity::getAuthorizeId, userId).or();
                queryWrapper.lambda().eq(SystemEntity::getAuthorizeId, PermissionConst.ALL_DEV_USER);
            }
        } else {
            queryWrapper.lambda().eq(SystemEntity::getUserId, userId);
        }
        return list(queryWrapper);
    }

    @Override
    public void workHandover(WorkHandoverModel workHandoverModel) {
        String userId = workHandoverModel.getFromId();
        List<String> appList = workHandoverModel.getAppList();
        String appHandoverUser = workHandoverModel.getAppHandoverUser();
        List<SystemEntity> listByIds = this.getListByIds(appList, null);
        for (SystemEntity entity : listByIds) {
            if (Objects.equals(entity.getUserId(), userId)) {
                entity.setUserId(appHandoverUser);
            }
            if (StringUtil.isNotEmpty(entity.getAuthorizeId())) {
                String[] userIds = entity.getAuthorizeId().split(",");
                String author = String.join(",", Arrays.stream(userIds).map(s -> s.equals(userId) ? appHandoverUser : s).toArray(String[]::new));
                entity.setAuthorizeId(author);
            }
            this.updateById(entity);
        }
    }

    @Override
    public void changeSystemAuthorizion(AppAuthorizationModel model) {
        String systemId = model.getSystemId();
        String createUserId = model.getCreateUserId();
        SystemEntity info = this.getInfo(systemId);

        if (StringUtil.isNotEmpty(createUserId)) {
            info.setUserId(createUserId);
        }
        this.updateById(info);
    }

    @Override
    public List<SystemEntity> getListByIds(List<String> list, List<String> moduleAuthorize, int type) {
        List<SystemEntity> systemList = new ArrayList<>(16);
        if (!list.isEmpty()) {
            QueryWrapper<SystemEntity> queryWrapper = new QueryWrapper<>();
            if (moduleAuthorize != null && !moduleAuthorize.isEmpty()) {
                queryWrapper.lambda().notIn(SystemEntity::getId, moduleAuthorize);
            }
            queryWrapper.lambda().in(SystemEntity::getId, list);
            if (type == 1) {
                queryWrapper.lambda().eq(SystemEntity::getEnabledMark, 1);
            }
            queryWrapper.lambda().orderByAsc(SystemEntity::getSortCode).orderByDesc(SystemEntity::getCreatorTime);
            return this.list(queryWrapper);
        }
        return systemList;
    }

    @Override
    public List<SystemEntity> getListByCreUser(String userId) {
        QueryWrapper<SystemEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(SystemEntity::getUserId, userId);
        queryWrapper.lambda().eq(SystemEntity::getEnabledMark, 1);
        return list(queryWrapper);
    }

}
