package jnpf.permission.controller;

import cn.dev33.satoken.annotation.SaCheckPermission;
import cn.dev33.satoken.annotation.SaMode;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.Parameters;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jnpf.base.ActionResult;
import jnpf.base.controller.SuperController;
import jnpf.base.service.DictionaryDataService;
import jnpf.base.service.SysconfigService;
import jnpf.base.util.ExcelTool;
import jnpf.base.vo.DownloadVO;
import jnpf.base.vo.ListVO;
import jnpf.base.vo.PageListVO;
import jnpf.base.vo.PaginationVO;
import jnpf.config.ConfigValueUtil;
import jnpf.constant.FileTypeConstant;
import jnpf.constant.MsgCode;
import jnpf.constant.PermissionConst;
import jnpf.emnus.OrgTypeEnum;
import jnpf.exception.DataException;
import jnpf.message.service.SynThirdDingTalkService;
import jnpf.message.service.SynThirdQyService;
import jnpf.message.util.SynThirdConsts;
import jnpf.model.*;
import jnpf.permission.constant.OrgColumnMap;
import jnpf.permission.entity.OrganizeEntity;
import jnpf.permission.entity.PositionEntity;
import jnpf.permission.model.organize.*;
import jnpf.permission.model.user.mod.UserIdModel;
import jnpf.permission.service.OrganizeService;
import jnpf.permission.service.PositionService;
import jnpf.permission.service.UserService;
import jnpf.util.*;
import jnpf.util.treeutil.SumTree;
import jnpf.util.treeutil.newtreeutil.TreeDotUtils;
import jnpf.workflow.service.TaskApi;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 组织机构
 * 组织架构：公司》部门》岗位》用户
 *
 * @author JNPF开发平台组
 * @version V3.1.0
 * @copyright 引迈信息技术有限公司
 * @date 2019年9月26日 上午9:18
 */
@Tag(name = "组织管理", description = "Organize")
@RestController
@RequestMapping("/api/permission/Organize")
@Slf4j
public class OrganizeController extends SuperController<OrganizeService, OrganizeEntity> {

    @Autowired
    private OrganizeService organizeService;
    @Autowired
    private SynThirdQyService synThirdQyService;
    @Autowired
    private SynThirdDingTalkService synThirdDingTalkService;
    @Autowired
    private SysconfigService sysconfigApi;
    @Autowired
    private PositionService positionService;
    @Autowired
    private TaskApi taskApi;

    @Operation(summary = "获取组织列表")
    @SaCheckPermission(value = {"permission.organize", "permission.user", "permission.role",
            "integrationCenter.dingTalk", "integrationCenter.weCom"}, mode = SaMode.OR)
    @Parameters({
            @Parameter(name = "id", description = "主键", required = true)
    })
    @GetMapping("/AsyncList/{id}")
    public ActionResult<PageListVO<OrganizeListVO>> getList(@PathVariable("id") String id, OrganizePagination pagination) {
        pagination.setParentId(id);
        pagination.setDataType(1);
        if (StringUtil.isNotEmpty(pagination.getKeyword())) {
            pagination.setParentId(null);
            pagination.setDataType(null);
        }
        List<OrganizeEntity> list = organizeService.getList(pagination);
        Map<String, Object> allOrgsTreeName = organizeService.getAllOrgsTreeName();
        List<OrganizeListVO> listVO = JsonUtil.getJsonToList(list, OrganizeListVO.class);
        for (OrganizeListVO item : listVO) {
            if (PermissionConst.DEPARTMENT.equals(item.getCategory())) {
                item.setIcon(PermissionConst.DEPARTMENT_ICON);
            } else {
                item.setIcon(PermissionConst.COMPANY_ICON);
            }
            if (StringUtil.isNotEmpty(item.getParentId()) && Objects.nonNull(allOrgsTreeName.get(item.getParentId()))) {
                item.setParentName(allOrgsTreeName.get(item.getParentId()).toString());
            }
            item.setHasChildren(true);
            String[] orgs = item.getOrganizeIdTree().split(",");
            item.setOrganizeIds(Arrays.asList(orgs));

        }
        ListVO<OrganizeListVO> vo = new ListVO<>();
        vo.setList(listVO);
        if (Objects.equals(pagination.getDataType(), 1)) {
            ActionResult.page(listVO, null);
        }
        PaginationVO jsonToBean = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        return ActionResult.page(listVO, jsonToBean);
    }

    @Operation(summary = "新建组织")
    @Parameters({
            @Parameter(name = "organizeCrForm", description = "组织模型", required = true)
    })
    @SaCheckPermission(value = {"permission.organize"})
    @PostMapping
    public ActionResult create(@RequestBody @Valid OrganizeCrForm organizeCrForm) {
        OrganizeEntity entity = JsonUtil.getJsonToBean(organizeCrForm, OrganizeEntity.class);
        if (organizeService.isExistByFullName(entity, false, false)) {
            return ActionResult.fail(MsgCode.EXIST001.get());
        }
        if (organizeService.isExistByEnCode(entity.getEnCode(), null)) {
            return ActionResult.fail(MsgCode.EXIST002.get());
        }
        //验证层级
        if (!organizeService.checkLevel(entity)) {
            return ActionResult.fail(MsgCode.PS036.get(sysconfigApi.getSysInfo().getOrgLevel()));
        }
        organizeService.create(entity);
        ThreadPoolExecutorUtil.getExecutor().execute(() -> {
            try {
                //创建组织后判断是否需要同步到企业微信
                synThirdQyService.unifyDepartmentSysToQy(false, entity, "", SynThirdConsts.CREAT_DEP);
                //创建组织后判断是否需要同步到钉钉
                synThirdDingTalkService.unifyDepartmentSysToDing(false, entity, "", SynThirdConsts.CREAT_DEP);
            } catch (Exception e) {
                log.error("创建组织后同步失败到企业微信或钉钉失败，异常：{}", e.getMessage());
            }
        });
        return ActionResult.success(MsgCode.SU001.get());
    }

    @Operation(summary = "更新组织")
    @Parameters({
            @Parameter(name = "id", description = "主键值", required = true),
            @Parameter(name = "organizeUpForm", description = "组织模型", required = true)
    })
    @SaCheckPermission(value = {"permission.organize"})
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid OrganizeUpForm organizeUpForm) {
        List<OrganizeEntity> synList = new ArrayList<>();
        OrganizeEntity entity = JsonUtil.getJsonToBean(organizeUpForm, OrganizeEntity.class);
        entity.setId(id);
        OrganizeEntity info = organizeService.getInfo(id);
        if (info == null) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        OrganizeEntity parentInfo = organizeService.getInfo(organizeUpForm.getParentId());
        //不能放到自己下级组织内
        if (id.equals(entity.getParentId()) || (parentInfo != null && parentInfo.getOrganizeIdTree() != null && parentInfo.getOrganizeIdTree().contains(id))) {
            return ActionResult.fail(MsgCode.SYS146.get(MsgCode.PS003.get()));
        }
        //todo 验证流程是否逐级审批
        if (!Objects.equals(info.getParentId(), organizeUpForm.getParentId())) {
            List<String> stepList = taskApi.getStepList();
            if (stepList.contains(id)) {
                return ActionResult.fail(MsgCode.OA029.get());
            }
        }

        entity.setId(id);
        if (organizeService.isExistByFullName(entity, false, true)) {
            return ActionResult.fail(MsgCode.EXIST001.get());
        }
        if (organizeService.isExistByEnCode(entity.getEnCode(), id)) {
            return ActionResult.fail(MsgCode.EXIST002.get());
        }
        //验证层级
        if (!organizeService.checkLevel(entity)) {
            return ActionResult.fail(MsgCode.PS036.get(sysconfigApi.getSysInfo().getOrgLevel()));
        }
        //验证下级类型
        if (!Objects.equals(info.getCategory(), entity.getCategory())) {
            if (!organizeService.checkOrgType(entity)) {
                return ActionResult.fail(MsgCode.PS037.get());
            }
        }
        boolean flag = organizeService.update(id, entity);
        synList.add(entity);

        // 父级id或者组织名称变化则更新下级所有树形信息
        if (!Objects.equals(entity.getParentId(), info.getParentId()) || !Objects.equals(entity.getFullName(), info.getFullName())) {
            List<String> underOrganizations = organizeService.getUnderOrganizations(id, false);
            underOrganizations.forEach(t -> {
                OrganizeEntity info1 = organizeService.getInfo(t);
                if (StringUtil.isNotEmpty(info1.getOrganizeIdTree())) {
                    organizeService.setOrgTreeIdAndName(info1);
                    organizeService.update(info1.getId(), info1);
                    synList.add(info1);
                }
            });
        }
        ThreadPoolExecutorUtil.getExecutor().execute(() -> {
            synList.forEach(t -> {
                try {
                    //修改组织后判断是否需要同步到企业微信
                    synThirdQyService.unifyDepartmentSysToQy(false, t, "",SynThirdConsts.UPDATE_DEP);
                    //修改组织后判断是否需要同步到钉钉
                    synThirdDingTalkService.unifyDepartmentSysToDing(false, t, "",SynThirdConsts.UPDATE_DEP);
                } catch (Exception e) {
                    log.error("修改组织后同步失败到企业微信或钉钉失败，异常：{}", e.getMessage());
                }
            });
        });
        if (!flag) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    @Operation(summary = "获取组织信息")
    @Parameters({
            @Parameter(name = "id", description = "主键值", required = true)
    })
    @SaCheckPermission(value = {"permission.organize"})
    @GetMapping("/{id}")
    public ActionResult<OrganizeInfoVO> info(@PathVariable("id") String id) throws DataException {
        OrganizeEntity entity = organizeService.getInfo(id);
        OrganizeInfoVO vo = JsonUtilEx.getJsonToBeanEx(entity, OrganizeInfoVO.class);
        if (StringUtil.isNotEmpty(entity.getParentId())) {
            OrganizeEntity pInfo = organizeService.getInfo(entity.getParentId());
            if (pInfo != null) {
                vo.setParentName(pInfo.getFullName());
                vo.setParentCategory(pInfo.getCategory());
            }
        }
        if (StringUtil.isNotEmpty(entity.getOrganizeIdTree())) {
            String replace = entity.getOrganizeIdTree().replace(entity.getId(), "");
            if (StringUtil.isNotEmpty(replace) && !",".equals(replace)) {
                vo.setOrganizeIdTree(Arrays.asList(replace.split(",")));
            } else {
                vo.setOrganizeIdTree(Collections.singletonList("-1"));
            }
        }
        return ActionResult.success(vo);
    }

    @Operation(summary = "删除组织")
    @Parameters({
            @Parameter(name = "id", description = "主键值", required = true)
    })
    @SaCheckPermission(value = {"permission.organize"})
    @DeleteMapping("/{id}")
    public ActionResult<String> delete(@PathVariable("id") String orgId) {
        return organizeService.delete(orgId, true);
    }

    //+++++++++++++++++++++++++++++++++++其他接口+++++++++++++++++++++++++++++

    @Operation(summary = "获取组织列表")
    @Parameters({
            @Parameter(name = "id", description = "主键", required = true)
    })
    @GetMapping("/Selector")
    public ActionResult<PageListVO<OrganizeSelectorVO>> getList(OrganizePagination pagination) {
        pagination.setDataType(1);
        if (StringUtil.isNotEmpty(pagination.getKeyword())) {
            pagination.setParentId(null);
            pagination.setDataType(null);
        }
        List<OrganizeEntity> list = organizeService.getList(pagination);
        Map<String, Object> allOrgsTreeName = organizeService.getAllOrgsTreeName();
        List<OrganizeSelectorVO> listVO = JsonUtil.getJsonToList(list, OrganizeSelectorVO.class);
        for (OrganizeSelectorVO item : listVO) {
            if (PermissionConst.DEPARTMENT.equals(item.getCategory())) {
                item.setIcon(PermissionConst.DEPARTMENT_ICON);
            } else {
                item.setIcon(PermissionConst.COMPANY_ICON);
            }
            if (StringUtil.isNotEmpty(pagination.getKeyword())) {
                item.setFullName(item.getOrgNameTree());
            } else {
                item.setHasChildren(true);
            }
            if (StringUtil.isNotEmpty(item.getParentId()) && Objects.nonNull(allOrgsTreeName.get(item.getParentId()))) {
                item.setParentName(allOrgsTreeName.get(item.getParentId()).toString());
            }
            String[] orgs = item.getOrganizeIdTree().split(",");
            item.setOrganizeIds(Arrays.asList(orgs));

        }
        PaginationVO jsonToBean = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        return ActionResult.page(listVO, jsonToBean);
    }

    @Operation(summary = "自定义范围回显")
    @Parameters({
            @Parameter(name = "userIdModel", description = "参数模型")
    })
    @PostMapping("/SelectedList")
    public ActionResult<ListVO<OrganizeListVO>> SelectedList(@RequestBody UserIdModel userIdModel) {
        List<OrganizeListVO> list = organizeService.selectedList(userIdModel.getIds());
        ListVO<OrganizeListVO> listVO = new ListVO<>();
        listVO.setList(list);
        return ActionResult.success(listVO);
    }

    @Operation(summary = "自定义范围下拉")
    @Parameters({
            @Parameter(name = "userIdModel", description = "参数模型")
    })
    @PostMapping("/OrganizeCondition")
    public ActionResult<ListVO<OrganizeListVO>> OrganizeCondition(@RequestBody UserIdModel userIdModel) {
        List<OrganizeEntity> list = organizeService.OrganizeCondition(userIdModel.getIds());
        List<OrganizeModel> modelList = JsonUtil.getJsonToList(list, OrganizeModel.class);
        for (OrganizeModel item : modelList) {
            if (PermissionConst.DEPARTMENT.equals(item.getCategory())) {
                item.setIcon(PermissionConst.DEPARTMENT_ICON);
            } else {
                item.setIcon(PermissionConst.COMPANY_ICON);
            }
        }
        List<SumTree<OrganizeModel>> trees = TreeDotUtils.convertListToTreeDot(modelList);
        List<OrganizeListVO> voList = JsonUtil.getJsonToList(trees, OrganizeListVO.class);
        for (OrganizeListVO item : voList) {
            item.setFullName(item.getOrgNameTree());
        }
        ListVO<OrganizeListVO> listVO = new ListVO<>();
        listVO.setList(voList);
        return ActionResult.success(listVO);
    }


    @Operation(summary = "组织异步带岗位")
    @Parameters({
            @Parameter(name = "id", description = "主键", required = true),
            @Parameter(name = "type", description = "类型：organize、position", required = true)
    })
    @GetMapping("/AsyncPosList")
    public ActionResult<PageListVO<OrganizePositonVO>> AsyncPosList(@RequestParam("id") String id, @RequestParam(name = "type") String type) {
        List<OrganizePositonVO> listVO = new ArrayList<>();
        //不是组织就是岗位
        type = StringUtil.isNotEmpty(type) && PermissionConst.ORGANIZE.equals(type) ? PermissionConst.ORGANIZE : PermissionConst.POSITION;
        Map<String, Object> allOrgsTreeName = organizeService.getAllOrgsTreeName();

        if (PermissionConst.ORGANIZE.equals(type)) {
            OrganizeEntity info = organizeService.getInfo(id);
            //添加岗位
            if (!Objects.equals(id, "0")) {
                String dutyPosition = info.getDutyPosition();
                List<PositionEntity> posList = positionService.getListByOrgIdOneLevel(Collections.singletonList(id));
                List<OrganizePositonVO> posListVo = JsonUtil.getJsonToList(posList, OrganizePositonVO.class);
                posListVo.forEach(item -> {
                    item.setType(PermissionConst.POSITION);
                    item.setIcon(PermissionConst.POSITION_ICON);
                    item.setHasChildren(true);
                    item.setOrganize(String.valueOf(allOrgsTreeName.get(item.getOrganizeId())));
                    item.setOrgNameTree(item.getOrganize() + "/" + item.getFullName());
                    if (StringUtil.isNotEmpty(dutyPosition) && dutyPosition.equals(item.getId())) {
                        item.setIsDutyPosition(1);
                    }
                });
                listVO.addAll(posListVo);
            }
            //添加组织
            OrganizePagination pagination = new OrganizePagination();
            pagination.setDataType(1);
            pagination.setParentId(id);
            List<OrganizeEntity> orgList = organizeService.getList(pagination);
            List<OrganizePositonVO> listOrgVO = JsonUtil.getJsonToList(orgList, OrganizePositonVO.class);
            for (OrganizePositonVO item : listOrgVO) {
                item.setType(PermissionConst.ORGANIZE);
                if (PermissionConst.DEPARTMENT.equals(item.getCategory())) {
                    item.setIcon(PermissionConst.DEPARTMENT_ICON);
                } else {
                    item.setIcon(PermissionConst.COMPANY_ICON);
                }
                item.setHasChildren(true);
            }
            listVO.addAll(listOrgVO);
        } else {
            List<PositionEntity> posList = positionService.getByParentId(id);
            listVO.addAll(JsonUtil.getJsonToList(posList, OrganizePositonVO.class));
            for (OrganizePositonVO item : listVO) {
                item.setType(PermissionConst.POSITION);
                item.setIcon(PermissionConst.POSITION_ICON);
                item.setHasChildren(true);
                item.setOrganize(String.valueOf(allOrgsTreeName.get(item.getOrganizeId())));
                item.setOrgNameTree(item.getOrganize() + "/" + item.getFullName());
            }
        }

        return ActionResult.page(listVO, null);
    }

    // -------------------导入导出--------------------

    @Operation(summary = "模板下载")
    @SaCheckPermission("permission.organize")
    @GetMapping("/TemplateDownload")
    public ActionResult<DownloadVO> TemplateDownload() {
        OrgColumnMap columnMap = new OrgColumnMap();
        String excelName = columnMap.getExcelName();
        Map<String, String> keyMap = columnMap.getColumnByType(0);
        List<ExcelColumnAttr> models = columnMap.getFieldsModel(false, 0);
        List<Map<String, Object>> list = columnMap.getDefaultList();
        Map<String, String[]> optionMap = getOptionMap();
        ExcelModel excelModel = ExcelModel.builder().models(models).selectKey(new ArrayList<>(keyMap.keySet())).optionMap(optionMap).build();
        DownloadVO vo = ExcelTool.getImportTemplate(FileTypeConstant.TEMPORARY, excelName, keyMap, list, excelModel);
        return ActionResult.success(vo);
    }


    @Operation(summary = "上传导入Excel")
    @SaCheckPermission("permission.organize")
    @PostMapping("/Uploader")
    public ActionResult<Object> Uploader() {
        return ExcelTool.uploader();
    }

    @Operation(summary = "导入预览")
    @SaCheckPermission("permission.organize")
    @GetMapping("/ImportPreview")
    public ActionResult<Map<String, Object>> ImportPreview(String fileName) throws Exception {
        // 导入字段
        OrgColumnMap columnMap = new OrgColumnMap();
        Map<String, String> keyMap = columnMap.getColumnByType(0);
        Map<String, Object> headAndDataMap = ExcelTool.importPreview(FileTypeConstant.TEMPORARY, fileName, keyMap);
        return ActionResult.success(headAndDataMap);
    }

    @Operation(summary = "导出异常报告")
    @SaCheckPermission("permission.organize")
    @PostMapping("/ExportExceptionData")
    public ActionResult<DownloadVO> ExportExceptionData(@RequestBody ExcelImportForm visualImportModel) {
        List<Map<String, Object>> dataList = visualImportModel.getList();
        OrgColumnMap columnMap = new OrgColumnMap();
        String excelName = columnMap.getExcelName();
        Map<String, String> keyMap = columnMap.getColumnByType(0);
        List<ExcelColumnAttr> models = columnMap.getFieldsModel(true, 0);
        ExcelModel excelModel = ExcelModel.builder().optionMap(getOptionMap()).models(models).build();
        DownloadVO vo = ExcelTool.exportExceptionReport(FileTypeConstant.TEMPORARY, excelName, keyMap, dataList, excelModel);
        return ActionResult.success(vo);
    }

    @Operation(summary = "导入数据")
    @SaCheckPermission("permission.organize")
    @PostMapping("/ImportData")
    public ActionResult<ExcelImportVO> ImportData(@RequestBody ExcelImportForm visualImportModel) throws Exception {
        List<Map<String, Object>> listData = new ArrayList<>();
        List<Map<String, Object>> headerRow = new ArrayList<>();
        if (visualImportModel.isType()) {
            ActionResult result = ImportPreview(visualImportModel.getFileName());
            if (result == null) {
                throw new Exception(MsgCode.FA018.get());
            }
            if (result.getCode() != 200) {
                throw new Exception(result.getMsg());
            }
            if (result.getData() instanceof Map) {
                Map<String, Object> data = (Map<String, Object>) result.getData();
                listData = (List<Map<String, Object>>) data.get("dataRow");
                headerRow = (List<Map<String, Object>>) data.get("headerRow");
            }
        } else {
            listData = visualImportModel.getList();
        }
        List<OrganizeEntity> addList = new ArrayList<>();
        List<Map<String, Object>> failList = new ArrayList<>();
        // 对数据做校验
        this.validateImportData(listData, addList, failList);

        //正常数据插入
        for (OrganizeEntity each : addList) {
            organizeService.create(each);
        }
        ExcelImportVO importModel = new ExcelImportVO();
        importModel.setSnum(addList.size());
        importModel.setFnum(failList.size());
        importModel.setResultType(!failList.isEmpty() ? 1 : 0);
        importModel.setFailResult(failList);
        importModel.setHeaderRow(headerRow);
        return ActionResult.success(importModel);
    }

    @Operation(summary = "导出Excel")
    @SaCheckPermission("permission.organize")
    @GetMapping("/ExportData")
    public ActionResult ExportData(OrganizePagination pagination) throws IOException {
        //获取当前用户的所有可见组织
        pagination.setDataType(1);
        List<OrganizeEntity> list = organizeService.getList(pagination);
        List<OrganizeEntity> allOrg = organizeService.getList(false);
        Map<String, String> idNameMap = allOrg.stream().collect(Collectors.toMap(OrganizeEntity::getId,
                e -> e.getFullName() + "/" + e.getEnCode(), (existing, replacement) -> existing));

        List<Map<String, Object>> realList = JsonUtil.getJsonToListMap(JsonUtil.getObjectToString(list));
        for (Map<String, Object> item : realList) {
            if (item.get("parentId") != null && !"-1".equals(item.get("parentId").toString())) {
                item.put("parentId", idNameMap.get(item.get("parentId")));
            } else {
                item.put("parentId", "");
            }
            item.put("category", OrgTypeEnum.get(item.get("category").toString()).getName());
        }

        //获取字段
        OrgColumnMap columnMap = new OrgColumnMap();
        String excelName = columnMap.getExcelName();
        Map<String, String> keyMap = columnMap.getColumnByType(0);
        List<ExcelColumnAttr> models = columnMap.getFieldsModel(true, 0);
        ExcelModel excelModel = ExcelModel.builder().models(models).selectKey(new ArrayList<>(keyMap.keySet())).build();
        DownloadVO vo = ExcelTool.creatModelExcel(FileTypeConstant.TEMPORARY, excelName, keyMap, realList, excelModel);
        return ActionResult.success(vo);
    }

    /**
     * 字段验证
     *
     * @param listData
     * @param addList
     * @param failList
     */
    private void validateImportData(List<Map<String, Object>> listData, List<OrganizeEntity> addList, List<Map<String, Object>> failList) {
        OrgColumnMap columnMap = new OrgColumnMap();
        Map<String, String> keyMap = columnMap.getColumnByType(0);
        Integer orgLevel = sysconfigApi.getSysInfo().getOrgLevel();

        List<OrganizeEntity> allOrg = organizeService.getList(false);
        Map<String, String> nameCodeMap = allOrg.stream().collect(Collectors.toMap(t -> t.getFullName() + "/" + t.getEnCode(), OrganizeEntity::getId));
        for (int i = 0, len = listData.size(); i < len; i++) {
            Map<String, Object> eachMap = listData.get(i);
            Map<String, Object> realMap = JsonUtil.getJsonToBean(eachMap, Map.class);
            StringJoiner errInfo = new StringJoiner(",");

            //处理父级组织
            String parentId;
            String organizeIdTree = "";
            List<String> childType = new ArrayList<>();
            if (eachMap.get("parentId") != null) {
                parentId = nameCodeMap.get(eachMap.get("parentId").toString());
                if (StringUtil.isEmpty(parentId)) {
                    errInfo.add("找不到上级组织");
                } else {
                    OrganizeEntity parentOrg = allOrg.stream().filter(t -> t.getId().equals(parentId)).findFirst().orElse(null);
                    if (parentOrg != null) {
                        childType = OrgTypeEnum.get(parentOrg.getCategory()).getChildType();
                        organizeIdTree = parentOrg.getOrganizeIdTree();
                    } else {
                        //库里不存在，从表格数据查询
                        parentOrg = addList.stream().filter(t -> t.getId().equals(parentId)).findFirst().orElse(null);
                        if (parentOrg != null) {
                            childType = OrgTypeEnum.get(parentOrg.getCategory()).getChildType();
                            organizeIdTree = parentOrg.getOrganizeIdTree() + "," + parentOrg.getId();
                        }
                    }
                    //层级限制
                    if (organizeIdTree.split(",").length > orgLevel) {
                        errInfo.add(MsgCode.PS036.get(orgLevel));
                    }
                }
            } else {
                parentId = "-1";
                childType = Arrays.asList(OrgTypeEnum.GROUP.getCode(), OrgTypeEnum.COMPANY.getCode(), OrgTypeEnum.AGENCY.getCode(),
                        OrgTypeEnum.OFFICE.getCode(), OrgTypeEnum.UNIT.getCode());
            }
            if (errInfo.length() > 0) {
                eachMap.put("errorsInfo", errInfo.toString());
                failList.add(eachMap);
                continue;
            }
            realMap.put("parentId", parentId);
            realMap.put("organizeIdTree", organizeIdTree);

            for (String column : keyMap.keySet()) {
                Object valueObj = eachMap.get(column);
                String value = valueObj == null ? null : String.valueOf(valueObj);
                String columnName = keyMap.get(column);
                switch (column) {
                    case "category":
                        String category = "";
                        if (eachMap.get("category") != null && StringUtil.isNotEmpty(eachMap.get("category").toString())) {
                            category = OrgTypeEnum.getByName(eachMap.get("category").toString()).getCode();
                        } else {
                            errInfo.add(columnName + "不能为空");
                            break;
                        }
                        //组织下级类型不支持
                        if (!childType.contains(category)) {
                            errInfo.add("组织类型异常，上级组织不允许添加该组织类型");
                            break;
                        }
                        realMap.put("category", category);
                        break;
                    case "fullName":
                        if (StringUtil.isEmpty(value)) {
                            errInfo.add(columnName + "不能为空");
                            break;
                        }
                        if (value.length() > 50) {
                            errInfo.add(columnName + "值超出最多输入字符限制");
                            break;
                        }

                        //库里判重
                        long fullNameCount = allOrg.stream().filter(t -> t.getParentId().equals(parentId) && t.getFullName().equals(value)).count();
                        if (fullNameCount > 0) {
                            errInfo.add(columnName + "值已存在");
                            break;
                        }
                        //表格内重复
                        fullNameCount = addList.stream().filter(t -> t.getParentId().equals(parentId) && t.getFullName().equals(value)).count();
                        if (fullNameCount > 0) {
                            errInfo.add(columnName + "值已存在");
                            break;
                        }
                        realMap.put("fullName", value);
                        break;
                    case "enCode":
                        if (StringUtil.isEmpty(value)) {
                            break;
                        }
                        if (value.length() > 50) {
                            errInfo.add(columnName + "值超出最多输入字符限制");
                            break;
                        }
                        if (!RegexUtils.checkEnCode(value)) {
                            errInfo.add(columnName + "只能输入英文、数字和小数点且小数点不能放在首尾");
                            break;
                        }
                        //库里重复
                        long enCodeCount = allOrg.stream().filter(t -> t.getEnCode().equals(value)).count();
                        if (enCodeCount > 0) {
                            errInfo.add(columnName + "值已存在");
                            break;
                        }
                        //表格内重复
                        enCodeCount = addList.stream().filter(t -> value.equals(t.getEnCode())).count();
                        if (enCodeCount > 0) {
                            errInfo.add(columnName + "值已存在");
                            break;
                        }
                        break;
                    case "sortCode":
                        if (StringUtil.isEmpty(value)) {
                            realMap.put("sortCode", 0);
                            break;
                        }
                        long numValue;
                        try {
                            numValue = Long.parseLong(value);
                        } catch (Exception e) {
                            errInfo.add(columnName + "值不正确");
                            break;
                        }
                        if (numValue < 0) {
                            errInfo.add(columnName + "值不能小于0");
                            break;
                        }
                        if (numValue > 1000000) {
                            errInfo.add(columnName + "值不能大于999999");
                            break;
                        }
                        break;
                    default:
                        break;
                }

            }
            if (errInfo.length() == 0) {
                OrganizeEntity organizeEntity = JsonUtil.getJsonToBean(realMap, OrganizeEntity.class);
                organizeEntity.setCreatorTime(new Date());
                String uuid = RandomUtil.uuId();
                organizeEntity.setId(uuid);
                if (StringUtil.isNotEmpty(organizeEntity.getEnCode())) {
                    nameCodeMap.put(organizeEntity.getFullName() + "/" + organizeEntity.getEnCode(), organizeEntity.getId());
                }
                addList.add(organizeEntity);
            } else {
                eachMap.put("errorsInfo", errInfo.toString());
                failList.add(eachMap);
            }
        }
    }

    /**
     * 获取下拉框
     *
     * @return
     */
    private Map<String, String[]> getOptionMap() {
        Map<String, String[]> optionMap = new HashMap<>();
        //类型
        String[] typeMap = Arrays.stream(OrgTypeEnum.values()).map(OrgTypeEnum::getName).toArray(String[]::new);
        optionMap.put("category", typeMap);
        return optionMap;
    }

}
