package jnpf.i18n.config;

import jnpf.config.ConfigValueUtil;
import jnpf.i18n.constant.I18nApiConst;
import jnpf.i18n.core.DynamicMessageSource;
import jnpf.i18n.core.MyReloadableResourceBundleMessageSource;
import jnpf.i18n.provider.*;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.context.MessageSourceProperties;
import org.springframework.context.MessageSource;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.context.support.ReloadableResourceBundleMessageSource;
import org.springframework.util.StringUtils;

import java.time.Duration;
import java.util.Collection;
import java.util.List;

/**
 * 国际化语言配置类
 */
@Configuration(proxyBeanMethods = false)
public class I18nAutoConfiguration {

    @Primary
    @Bean(AbstractApplicationContext.MESSAGE_SOURCE_BEAN_NAME)
    public MessageSource messageSource(MessageSourceProperties properties, List<MessageSourceProvider> messageSourceProviders, DynamicMessageSource dynamicMessageSource) {
        ReloadableResourceBundleMessageSource messageSource = new MyReloadableResourceBundleMessageSource(messageSourceProviders);
        Object basename = properties.getBasename();
        if (basename != null) {
            if(basename instanceof String) {
                if (!((String) basename).isEmpty()) {
                    messageSource.setBasenames(StringUtils
                            .commaDelimitedListToStringArray(StringUtils.trimAllWhitespace((String) basename)));
                }
            }else if(basename instanceof Collection){
                if(!((Collection<?>) basename).isEmpty()){
                    messageSource.setBasenames(((Collection<?>) basename).toArray(new String[0]));
                }
            }
        }
        if (properties.getEncoding() != null) {
            messageSource.setDefaultEncoding(properties.getEncoding().name());
        }
        messageSource.setFallbackToSystemLocale(properties.isFallbackToSystemLocale());
        Duration cacheDuration = properties.getCacheDuration();
        if (cacheDuration != null) {
            messageSource.setCacheMillis(cacheDuration.toMillis());
        }
        messageSource.setAlwaysUseMessageFormat(properties.isAlwaysUseMessageFormat());
        messageSource.setUseCodeAsDefaultMessage(properties.isUseCodeAsDefaultMessage());

        dynamicMessageSource.setParentMessageSource(messageSource);
        return dynamicMessageSource;
    }

    @Bean
    @ConditionalOnMissingBean
    public DynamicMessageSource getDynamicMessageSource(I18nMessageProvider i18nMessageProvider){
        return new DynamicMessageSource(i18nMessageProvider);
    }

    @Bean
    @ConditionalOnMissingBean
    public DynamicMessageProvider getDynamicMessageProvider(){
        return new MyDynamicMessageProvider();
    }

    @Bean
    @ConditionalOnMissingBean
    public I18nMessageProvider getI18nMessageProvider(ConfigValueUtil configValueUtil, DynamicMessageProvider dynamicMessageProvider){
        return new MyI18nMessageProvider(configValueUtil, dynamicMessageProvider);
    }

    @Bean
    public I18nApiConst getI18nApiConst(ConfigValueUtil configValueUtil){
        return new I18nApiConst(configValueUtil.getApiDomain());
    }


}
